﻿using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Configuration;
using System.Windows.Forms;
using System.Web;
using System.Net;
using System.Net.Security;
using System.Security.Cryptography.X509Certificates;
using NConsoler;
using Snowball.Common;
using Newtonsoft.Json;

namespace SynoGet
{
    class SynoGet
    {
        // some defaults
        static string ds_protocol       = "https";
        static string ds_ip             = "";
        static string ds_port           = "5001";
        static string ds_path_login     = "webman/login.cgi";
        static string ds_path_download  = "webman/3rdparty/DownloadStation/dlm/downloadman.cgi";
        static string ds_username       = "";
        static string ds_password       = "";
        static bool debug = false;

        /// <summary>
        /// Main entrypoint.
        /// </summary>
        /// <param name="args">Commandline args (parsed by NConsoler).</param>
        static void Main(string[] args)
        {
            // redirect the console to a memory stream (if errors, write output in a file later)
            MemoryStream memStream = new MemoryStream();
            TextWriter streamWrite = new StreamWriter(memStream);
            TextReader streamRead = new StreamReader(memStream);
            ((StreamWriter)streamWrite).AutoFlush = true;
            TextWriter origConsole = Console.Out;
            Console.SetOut(streamWrite);

            // read config from SynoGet.exe.config
            AppSettingsReader config = new AppSettingsReader();
            SynoGet.ds_protocol         = readConfigString(SynoGet.ds_protocol,         "ds_protocol",          config);
            SynoGet.ds_ip               = readConfigString(SynoGet.ds_ip,               "ds_ip",                config);
            SynoGet.ds_port             = readConfigString(SynoGet.ds_port,             "ds_port",              config);
            SynoGet.ds_path_login       = readConfigString(SynoGet.ds_path_login,       "ds_path_login",        config);
            SynoGet.ds_path_download    = readConfigString(SynoGet.ds_path_download,    "ds_path_download",     config);
            SynoGet.ds_username         = readConfigString(SynoGet.ds_username,         "ds_username",          config);
            SynoGet.ds_password         = readConfigString(SynoGet.ds_password,         "ds_password",          config);

            // show help?
            if (args.Length == 0)
            {
                HelpWindow hw = new HelpWindow();
                hw.ShowDialog();
                return;
            }

            // run
            Consolery.Run(typeof(SynoGet), args);

            try
            {
                Console.SetOut(origConsole);
                memStream.Position = 0;
                Console.WriteLine(streamRead.ReadToEnd());
                memStream.Position = 0;
                string errorString = streamRead.ReadToEnd();
                if (debug || (errorString != null && errorString != ""))
                {
                    string filename = "SynoGetErrors.txt";
                    if (debug) filename = "SynoGetDebug.txt";
                    StreamWriter fileWriter = new StreamWriter(filename, true);
                    fileWriter.WriteLine(DateTime.Now.ToLocalTime().ToString());
                    fileWriter.Write(errorString);
                    fileWriter.WriteLine();
                    fileWriter.Flush();
                    fileWriter.Close();
                }
            }
            catch { }
        }

        /// <summary>
        /// Reads the config from SynoGet.exe.config, if existent, sets a default string else.
        /// </summary>
        /// <param name="defaultValue">Set, if no entry in config.</param>
        /// <param name="configKey">The key, that should be read from the config.</param>
        /// <param name="config">The config reader object.</param>
        /// <returns></returns>
        public static string readConfigString(string defaultValue, string configKey, AppSettingsReader config)
        {
            string outString = defaultValue;
            try
            {
                outString = (string)config.GetValue(configKey, typeof(string));
            }
            catch { }
            return outString;
        }

        /// <summary>
        /// Does all the stuff to add a download to the diskstation.
        /// </summary>
        /// <param name="downloadurl">The URL to download.</param>
        /// <param name="ds_protocol">The used protocoll of the diskstation webman (http or https).</param>
        /// <param name="ds_ip">The IP adress (or DNS name) of the diskstation.</param>
        /// <param name="ds_port">The port of the webman of the diskstation.</param>
        /// <param name="ds_path_login">The path of the login.cgi</param>
        /// <param name="ds_path_download">The path of the downloadman.cgi</param>
        /// <param name="ds_username">The username of the diskstation user.</param>
        /// <param name="ds_password">The password of the diskstation user.</param>
        /// <param name="passwordinplaintext">Switches to unencrypted passwords.</param>
        [Action]
        public static void AddDownload
            (
            [Required]     string downloadurl,
            [Optional("")] string ds_protocol,
            [Optional("")] string ds_ip,
            [Optional("")] string ds_port,
            [Optional("")] string ds_path_login,
            [Optional("")] string ds_path_download,
            [Optional("")] string ds_username,
            [Optional("")] string ds_password,
            [Optional(false)] bool passwordinplaintext,
            [Optional(false)] bool debug
            )
        {
            // set defaults if not specified by commandline
            ds_protocol         = (ds_protocol          == "") ? SynoGet.ds_protocol        : ds_protocol;
            ds_ip               = (ds_ip                == "") ? SynoGet.ds_ip              : ds_ip;
            ds_port             = (ds_port              == "") ? SynoGet.ds_port            : ds_port;
            ds_path_login       = (ds_path_login        == "") ? SynoGet.ds_path_login      : ds_path_login;
            ds_path_download    = (ds_path_download     == "") ? SynoGet.ds_path_download   : ds_path_download;
            ds_username         = (ds_username          == "") ? SynoGet.ds_username        : ds_username;
            ds_password         = (ds_password          == "") ? SynoGet.ds_password        : ds_password;
            
            SynoGet.debug = debug;

            // print parameters
            if (debug)
            {
                Console.WriteLine("Parameters: ");
                Console.WriteLine("downloadurl          -> " + downloadurl);
                Console.WriteLine("ds_protocol          -> " + ds_protocol);
                Console.WriteLine("ds_ip                -> " + ds_ip);
                Console.WriteLine("ds_port              -> " + ds_port);
                Console.WriteLine("ds_path_login        -> " + ds_path_login);
                Console.WriteLine("ds_path_download     -> " + ds_path_download);
                Console.WriteLine("ds_username          -> " + ds_username);
                Console.WriteLine("ds_password          -> " + ds_password);
                Console.WriteLine("passwordinplaintext  -> " + passwordinplaintext.ToString());
                Console.WriteLine("debug                -> " + debug.ToString());
                Console.WriteLine();
            }

            // decode (base64) password
            if (!passwordinplaintext)
            {
                if (debug) Console.WriteLine("Password not in plaintext. Decoding password base64...");

                ds_password = SynoGet.decodePassword(ds_password);

                if (debug) Console.WriteLine("Password decoded. Plaintext password: " + ds_password);
                if (debug) Console.WriteLine();
            }

            // is the ip of the diskstation given?
            if (ds_ip == null) 
            {
                Console.WriteLine("Missing IP or Name of the diskstation (/ds_ip:<ip> or SynoGet.exe.config). Aborting.");
                return;
            }


            if (debug) Console.WriteLine("Constructing full URL of diskstation login.cgi...");
            string ds_loginurl = ds_protocol + "://" + ds_ip + ":" + ds_port + "/" + ds_path_login;

            if (debug) Console.WriteLine("Full URL of login.cgi:");
            if (debug) Console.WriteLine(ds_loginurl);
            if (debug) Console.WriteLine();


            // accept all https certificates
            if (debug) Console.WriteLine("Configuring to accept all https certificates...");
            System.Net.ServicePointManager.ServerCertificateValidationCallback += new System.Net.Security.RemoteCertificateValidationCallback(ValidateCertificate);
            if (debug) Console.WriteLine("Now all https certificates will be accepted.");
            if (debug) Console.WriteLine();

            // post the login request
            if (debug) Console.WriteLine("Constructing the HTTP-Post for login...");
            if (debug) Console.WriteLine("Instantiating the PostSubmitter object...");
            PostSubmitter loginPost = new PostSubmitter();
            if (debug) Console.WriteLine("PostSubmitter object instantiated.");

            if (debug) Console.WriteLine("Setting URL to " + ds_loginurl);
            loginPost.Url = ds_loginurl;

            if (debug) Console.WriteLine("Adding post item: username    -> " + ds_username);
            loginPost.PostItems.Add("username", ds_username);

            if (debug) Console.WriteLine("Adding post item: passwd      -> " + ds_password);
            loginPost.PostItems.Add("passwd", ds_password);

            if (debug) Console.WriteLine("Setting PostSubmitter type to POST...");
            loginPost.Type = PostSubmitter.PostTypeEnum.Post;

            if (debug) Console.WriteLine("Executing the HTTP-Post...");
            string loginResponseString = loginPost.Post();
            if (debug) Console.WriteLine("Successfully executed the HTTP-Post.");
            if (debug) Console.WriteLine();

            if (debug) Console.WriteLine("Getting the cookie from the HTTP-Post-request...");
            CookieContainer cc = loginPost.CookieContainer;
            if (debug) Console.WriteLine("Successfully acquired the HTTP-Post-request cookie.");
            if (debug)
            {
                try
                {
                    if (cc.Count > 0)
                    {
                        Cookie firstCookie = cc.GetCookies(new Uri(ds_protocol + "://" + ds_ip))[0];
                        Console.WriteLine("Cookie-Domain: " + firstCookie.Domain);
                        Console.WriteLine("Cookie-Path  : " + firstCookie.Path);
                        Console.WriteLine("Cookie-Name  : " + firstCookie.Name);
                        Console.WriteLine("Cookie-Value : " + firstCookie.Value);
                    }
                }
                catch { }
            }
            if (debug) Console.WriteLine();

            if (debug) Console.WriteLine("Deserializing JSON response object...");
            JSONLoginResponse loginResponse = JavaScriptConvert.DeserializeObject<JSONLoginResponse>(loginResponseString);
            
            if (loginResponse.success && loginResponse.result.Equals("success"))
            {
                if (debug) Console.WriteLine("Deserialization successful.");
                if (debug) Console.WriteLine();

                // post download request
                if (debug) Console.WriteLine("Constructing the HTTP-Post for enqueueing the download...");
                if (debug) Console.WriteLine("Instantiating the PostSubmitter object...");
                PostSubmitter downloadEnqueuePost = new PostSubmitter();
                if (debug) Console.WriteLine("PostSubmitter object instantiated.");

                if (debug) Console.WriteLine("Constructing full URL of diskstation downloadman.cgi...");
                string ds_downloadurl = ds_protocol + "://" + ds_ip + ":" + ds_port + "/" + ds_path_download;

                if (debug) Console.WriteLine("Setting URL to " + ds_downloadurl);
                downloadEnqueuePost.Url = ds_downloadurl;

                if (debug) Console.WriteLine("Adding post item: action      -> add_url_task");
                downloadEnqueuePost.PostItems.Add("action", "add_url_task");

                if (debug) Console.WriteLine("Adding post item: urls        -> [\"" + downloadurl + "\"]");
                downloadEnqueuePost.PostItems.Add("urls", "[\"" + downloadurl + "\"]");

                if (debug) Console.WriteLine("Setting PostSubmitter type to POST");
                downloadEnqueuePost.Type = PostSubmitter.PostTypeEnum.Post;

                if (debug) Console.WriteLine("Setting the cookie of the HTTP-Post-request...");
                downloadEnqueuePost.CookieContainer = cc;
                if (debug) Console.WriteLine("Successfully set the HTTP-Post-request cookie.");

                if (debug) Console.WriteLine("Executing the HTTP-Post...");
                string downloadEnqueueResponseString = downloadEnqueuePost.Post();
                if (debug) Console.WriteLine("Successfully executed the HTTP-Post.");
                if (debug) Console.WriteLine();

                if (debug) Console.WriteLine("Deserializing JSON response object...");
                JSONDownloadEnqueueResponse downloadEnqueueResponse = JavaScriptConvert.DeserializeObject<JSONDownloadEnqueueResponse>(downloadEnqueueResponseString);

                if (!downloadEnqueueResponse.success)
                {
                    // download enqueue not successful
                    Console.WriteLine("Enqueue of download NOT successful.");
                }
                else
                {
                    if (debug) Console.WriteLine("Download queued.");
                    if (debug) Console.WriteLine("Exiting.");
                    if (debug) Console.WriteLine();
                }
            }
            else
            {
                // login not successful
                Console.WriteLine("Login NOT successful.");
            }

        }


        private static bool ValidateCertificate(object sender, X509Certificate certificate, X509Chain chain, SslPolicyErrors sslPolicyErrors)
        {
            return true;
        }

        private static string decodePassword(string codedPassword)
        {
            string decodedPassword = codedPassword;
            try
            {
                byte[] decoded_byte = Convert.FromBase64String(codedPassword);
                decodedPassword = System.Text.Encoding.UTF8.GetString(decoded_byte);
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            return decodedPassword;
        }
    }
}
